------------------------------------------------------------------------
-- Event:        Delphi Day 2018, Piacenza, June 06 2018               -
--               https://www.delphiday.it/                             -
-- Seminary:     How to write high performance queries in T-SQL        -
-- Demo:         OVER Clause                                           -
-- Author:       Sergio Govoni                                         -
-- Notes:        --                                                    -
------------------------------------------------------------------------

USE [WideWorldImporters];
GO

SET STATISTICS IO ON;
GO

-- View the data
SELECT * FROM Warehouse.StockItemTransactions;
GO

/*
DBCC FREEPROCCACHE;
GO
*/


------------------------------------------------------------------------
-- How high are the products sales?                                    -
------------------------------------------------------------------------

-- Traditional solution with multiple CTE

-- I need a grouping quantity that is the grand total,
-- but when I do a grouping I miss the detail!

WITH SalesByProduct AS
(
  -- Calculate the total sales quantity by product
  SELECT
    StockItemID
	,SUM(Quantity) AS TotalQtyByProduct
  FROM
    Warehouse.StockItemTransactions
  WHERE
    (InvoiceID IS NOT NULL)
    AND (CustomerID IS NOT NULL)
  GROUP BY StockItemID
),
SalesAll AS
(
  -- Calculate the grand total sales quantity
  SELECT
    SUM(Quantity) AS GrandTotalQty
  FROM
    Warehouse.StockItemTransactions
  WHERE
    (InvoiceID IS NOT NULL)
    AND (CustomerID IS NOT NULL)
)
-- Join the Product table to the first CTE
-- and by the CROSS JOIN join the second CTE
SELECT
  P.StockItemID
  ,P.StockItemName
  ,SBP.TotalQtyByProduct
  ,SAll.GrandTotalQty
  ,((SBP.TotalQtyByProduct / SAll.GrandTotalQty) * 100.) AS [%]
FROM
  Warehouse.StockItems AS P
JOIN
  SalesByProduct AS SBP ON SBP.StockItemID=P.StockItemID
CROSS JOIN
  SalesAll AS SALL
ORDER BY
  [%] DESC;
GO



-- Traditional solution with Sub-queries
-- Each sub-query requires separate scan

SELECT
  P.StockItemID
  ,P.StockItemName
  ,SUM(TH.Quantity) AS TotalQtyByProduct
  ,(
     -- Calculation of Grand Total Sales Quantity
     SELECT
       SUM(Quantity)
	 FROM
	   Warehouse.StockItemTransactions  -- <== 3
     WHERE
	   (InvoiceID IS NOT NULL)
       AND (CustomerID IS NOT NULL)
   ) AS GrandTotalQty

  ,(SUM(TH.Quantity) / 
     (
       -- Calculation of Grand Total Sales Quantity
	   SELECT
         SUM(Quantity)
	   FROM
	     Warehouse.StockItemTransactions  -- <== 2
	   WHERE
	     (InvoiceID IS NOT NULL)
         AND (CustomerID IS NOT NULL)
	 )
   ) * 100 AS [%]
FROM
  Warehouse.StockItems AS P
JOIN
  Warehouse.StockItemTransactions AS TH ON P.StockItemID=TH.StockItemID  -- <== 1
WHERE
  (TH.InvoiceID IS NOT NULL)
  AND (TH.CustomerID IS NOT NULL)
GROUP BY
  P.StockItemID
  ,P.StockItemName
ORDER BY
  [%] DESC;
GO








-- OVER clause solution
-- These calculation have been done without groupings
-- and without missing details!
WITH CTE AS
(
  SELECT
    TH.StockItemID
    -- #1 Calculate the number of pieces sold for each Product
    ,SUM(TH.Quantity) OVER (PARTITION BY TH.StockItemID) AS TotalQtyByProduct
    -- #2 Calculate the grand total quantity of Products sold
    ,SUM(TH.Quantity) OVER () AS GrandTotalQty
    ,RNumb = ROW_NUMBER() OVER (PARTITION BY TH.StockItemID ORDER BY TH.StockItemID)
  FROM
    Warehouse.StockItemTransactions AS TH
  WHERE
    (TH.InvoiceID IS NOT NULL)
    AND (TH.CustomerID IS NOT NULL)
)
SELECT
  P.StockItemID
  ,P.StockItemName
  ,[%] = (CTE.TotalQtyByProduct / CTE.GrandTotalQty * 100.)
FROM
  CTE
JOIN
  Warehouse.StockItems AS P ON P.StockItemID=CTE.StockItemID
WHERE
  (RNumb = 1)
ORDER BY
  [%] DESC


/*
SELECT
  DISTINCT
  TH.StockItemID
  ,P.StockItemName
  ,(
     -- #1 Calculate the number of pieces sold for each Product
     SUM(TH.Quantity) OVER (PARTITION BY TH.StockItemID) / 
     -- #2 Calculate the grand total quantity of Products sold
	 SUM(TH.Quantity) OVER ()
   ) * 100. AS [%]
FROM
  Warehouse.StockItemTransactions AS TH
JOIN
  Warehouse.StockItems AS P ON P.StockItemID=TH.StockItemID
WHERE
  (TH.InvoiceID IS NOT NULL)
  AND (TH.CustomerID IS NOT NULL)
ORDER BY
  [%] DESC;
GO
*/





------------------------------------------------------------------------
-- Dynamic product stock level in a warehouse                          -
------------------------------------------------------------------------

-- Running total

-- Each row in the TransactionHistory table represents a transaction in
-- a warehouse

-- TransactionType
-- W = WorkOrder, S = SalesOrder, P = PurchaseOrder

-- When the transaction has a TransactionType equal to "W" or "P" the
-- transaction has positive quantity, when it has a TransactionType equal to "S"
-- the transaction has negative quantity

/*
StockItemID StockItemTransactionID Quantity       StockLevel  
----------- ---------------------- -------------- -----------------------
80          138                    24.000         24.000    (24)
80          353                    -12.000        12.000    (= 24 - 12)
80          499                    -84.000        -72.000   (= 12 - 84)
80          744                    -12.000        -84.000   (= -12 - 72)
80          1149                   -72.000        -156.000
80          1445                   -60.000        -216.000
80          1480                   -72.000        -288.000
80          1672                   -36.000        -324.000
80          1726                   -24.000        -348.000
80          1784                   -96.000        -444.000
80          2334                   -96.000        -540.000
80          3349                   -12.000        -552.000
80          3661                   -120.000       -672.000
80          3940                   -96.000        -768.000
80          4280                   -72.000        -840.000
*/


-- Traditional set-based solution with joins

-- Using a join you can calculate the running total by filtering
-- in the second instance all rows that have the same ProductID
-- and TransactionID that is less than or equal to the one
-- in the first instance

/*
SELECT
 NRow = COUNT(StockItemID)
 ,StockItemID
FROM
  Warehouse.StockItemTransactions
GROUP BY
  StockItemID
ORDER BY 1 ASC;
GO
*/


SELECT
  T.StockItemID
  ,T.StockItemTransactionID
  ,T.TransactionOccurredWhen
  ,T.TransactionTypeID
  ,T.Quantity
  ,SUM(T1.Quantity) AS StockLevel
FROM
  Warehouse.StockItemTransactions AS T
JOIN
  Warehouse.StockItemTransactions AS T1 ON
    (T1.StockItemID = T.StockItemID)
	AND (T1.StockItemTransactionID <= T.StockItemTransactionID)
WHERE
  (T.StockItemID = 80)
GROUP BY
  T.StockItemID
  ,T.TransactionOccurredWhen
  ,T.TransactionTypeID
  ,T.Quantity
  ,T.StockItemTransactionID
ORDER BY
  T.StockItemID
  ,T.StockItemTransactionID;
GO





-- For the N rows of some ProductID in the instance named T
-- it will find N matches in the instance T1 with rows 1 up to N

-- For each partition, the plan scans
-- 1 + 2 + 3 + ... + R rows which is equal to (R + R^2)/2

-- It is an arithmetic sequence!

select (10000 + 10000*10000)/2

-- Without the filter "StockItemID = 80" the number of rows
-- processed in these plans is P*R + P(R + R^2)/2

-- This plan has square complexity !!





-- If you increase the rows in the partition of a factor T
-- the complexity will be increases by T^2


-- Traditional solution with sub-queries

SELECT
  T.StockItemID
  ,T.TransactionOccurredWhen
  ,T.TransactionTypeID
  ,T.Quantity
  ,(
     SELECT
	   SUM(T1.Quantity)
	 FROM
	   Warehouse.StockItemTransactions AS T1
	 WHERE
	   (T.StockItemID = T1.StockItemID)
	   AND (T1.StockItemTransactionID <= T.StockItemTransactionID)
   ) AS StockLevel
FROM
  Warehouse.StockItemTransactions AS T
WHERE
  (T.StockItemID = 80)
ORDER BY
  T.StockItemID
  ,T.StockItemTransactionID;
GO






-- Cursor-based solution

DBCC FREEPROCCACHE;
GO


-- !! Turn OFF the STATISTICS IO !!

SET STATISTICS IO OFF;
GO


-- !! Turn OFF the Execution Plan !!




-- Cursor-based solution has linear complexity,
-- but it is very verbosity!

BEGIN
  DECLARE
    @StockLevelTab AS TABLE
    (
      ProductID INTEGER NOT NULL
	    ,TransactionID INTEGER NOT NULL
  	  ,TransactionTypeID INTEGER NOT NULL
  	  ,TransactionDate DATETIME NOT NULL
      ,Quantity INTEGER NOT NULL
      ,StockLevel INTEGER NOT NULL
    );

  DECLARE
    @ProductID INTEGER
    ,@TransactionID INTEGER
    ,@PrevProductID INTEGER
    ,@Quantity INTEGER
    ,@StockLevel BIGINT
    ,@TransactionDate DATETIME
    ,@TransactionTypeID INTEGER;


  -- Declare cursor
  DECLARE StockByProduct CURSOR LOCAL FAST_FORWARD FOR
    SELECT
      StockItemID
	  ,StockItemTransactionID
	  ,Quantity
	  ,TransactionOccurredWhen
	  ,TransactionTypeID
    FROM
      Warehouse.StockItemTransactionsNoCCI
	WHERE
	  (StockItemID = 80)
    ORDER BY
      StockItemID
	  ,StockItemTransactionID;

  OPEN StockByProduct;

  FETCH NEXT FROM StockByProduct
    INTO @ProductID
	     ,@TransactionID
		 ,@Quantity
		 ,@TransactionDate
		 ,@TransactionTypeID;

  SELECT @PrevProductID = @ProductID, @StockLevel = 0;

  WHILE (@@FETCH_STATUS = 0)
  BEGIN
    IF (@PrevProductID <> @ProductID)
	  SELECT @PrevProductID = @ProductID, @StockLevel = 0;

    SET @StockLevel = @StockLevel + @Quantity;

    INSERT INTO @StockLevelTab
	(
	  ProductID
	  ,TransactionID
	  ,TransactionDate
	  ,TransactionTypeID
	  ,Quantity
	  ,StockLevel
	)
	VALUES
	(
	  @ProductID
	  ,@TransactionID
	  ,@TransactionDate
	  ,@TransactionTypeID
	  ,@Quantity
	  ,@StockLevel
	);

    FETCH NEXT FROM StockByProduct
	  INTO @ProductID
	       ,@TransactionID
		   ,@Quantity
		   ,@TransactionDate
		   ,@TransactionTypeID;
  END

  CLOSE StockByProduct;

  DEALLOCATE StockByProduct;

  SELECT
    *
  FROM
    @StockLevelTab
  ORDER BY
    ProductID
	,TransactionID;
END;
GO




-- Running total in SQL Server (2012+)

-- The window specification is intuitive here, we need to partition
-- the window by ProductID ordering by TransactionID

-- Supposing that I'm in a certain row, the dynamic products stock level
-- is the SUM of the quantity of this particular row up to the first one,
-- this is the filter frame

SELECT
  StockItemID
  ,StockItemTransactionID
  ,TransactionTypeID
  ,TransactionOccurredWhen
  ,Quantity
  ,StockLevel = SUM(Quantity)
                OVER (
				      -- Partition the window by ProductID
				      PARTITION BY			        
				        StockItemID
              -- Ordering by TransactionID
	                  ORDER BY
					    StockItemTransactionID
              -- This particular row up to the first one
                      ROWS
					    BETWEEN UNBOUNDED PRECEDING
		                AND CURRENT ROW
					 )
FROM
  Warehouse.StockItemTransactions
--WHERE
--  (StockItemID = 80)
ORDER BY
  StockItemID
  ,StockItemTransactionID;
GO





------------------------------------------------------------------------
-- Windowing functions                                                 -
-- LEAD, LAG                                                           -
-- LAST_VALUE, FIRST_VALUE                                             -
------------------------------------------------------------------------

-- LEAD is able to access to "X" following rows

-- Number of days elapsed between an order and the following one
-- by the same customer
WITH CTE AS
(
  SELECT
    DISTINCT
    CustomerID
    ,OrderDate AS cur_d
    ,next_d = LEAD(OrderDate, 1, GETDATE()) OVER (PARTITION BY CustomerID
                                                  ORDER BY CustomerID, OrderDate)
  FROM
    Sales.Orders
)
SELECT
  CustomerID
  ,cur_d
  ,next_d
  ,dif_day  = DATEDIFF(DAY, cur_d, next_d)
  ,avg_cust = AVG(DATEDIFF(DAY, cur_d, next_d)) OVER (PARTITION BY CustomerID)
  ,avg_all  = AVG(DATEDIFF(DAY, cur_d, next_d)) OVER ()
FROM
  CTE
ORDER BY
  CustomerID, cur_d
GO



SELECT * FROM Purchasing.PurchaseOrders;
GO

SELECT
  SupplierID
  ,PurchaseOrderID
  ,OrderDate
  ,[OrderDate +1] = LEAD(OrderDate, 1, NULL)
                      OVER (PARTITION BY SupplierID ORDER BY OrderDate)
  ,[OrderDate +5] = LEAD(OrderDate, 5, NULL)
                      OVER (PARTITION BY SupplierID ORDER BY OrderDate)
FROM
  Purchasing.PurchaseOrders;
GO




-- LAG is able to access to "N" previous rows
SELECT
  SupplierID
  ,PurchaseOrderID
  ,OrderDate
  ,[OrderDate -1] = LAG(OrderDate)
                      OVER (PARTITION BY SupplierID ORDER BY OrderDate)
  ,[OrderDate -5] = LAG(OrderDate, 5, NULL)
                      OVER (PARTITION BY SupplierID ORDER BY OrderDate)
FROM
  Purchasing.PurchaseOrders;
GO




-- FIRST_VALUE is able to access to the first value of the data window
-- LAST_VALUE is able to access to the last value of the data window
SELECT
  SupplierID
  ,PurchaseOrderID
  ,OrderDate
  ,FIRST_VALUE(OrderDate) OVER (PARTITION BY
                                  SupplierID
                                ORDER BY
                                  OrderDate) AS [FIRST OrderDate]
  ,LAST_VALUE(OrderDate) OVER (PARTITION BY
                                 SupplierID
                               ORDER BY
                                 OrderDate
                               ROWS BETWEEN UNBOUNDED PRECEDING AND
                                            UNBOUNDED FOLLOWING) AS [LAST OrderDate]
FROM
  Purchasing.PurchaseOrders;
GO